;;;; HSV Parser Tests (Common Lisp)

(load "hsv.lisp")
(in-package :hsv)

(defvar *passed* 0)
(defvar *failed* 0)

(defun assert-true (condition name)
  (if condition
      (progn
        (format t "✓ ~A~%" name)
        (incf *passed*))
      (progn
        (format t "✗ ~A~%" name)
        (incf *failed*))))

(defun assert-eq (got expected name)
  (assert-true (equal got expected)
               (format nil "~A (got: ~S, expected: ~S)" name got expected)))

(defun test-basic ()
  (let* ((input (format nil "~Cname~CAlice~Cage~C30~C" +stx+ +us+ +rs+ +us+ +etx+))
         (doc (parse input)))
    (assert-eq (length (document-records doc)) 1 "Basic: 1 record")
    (assert-eq (get-value (first (document-records doc)) "name") "Alice" "Basic: name=Alice")
    (assert-eq (get-value (first (document-records doc)) "age") "30" "Basic: age=30")))

(defun test-multiple-records ()
  (let* ((input (format nil "~Cname~CAlice~Cname~CBob~C" +stx+ +us+ +fs+ +us+ +etx+))
         (doc (parse input)))
    (assert-eq (length (document-records doc)) 2 "Multiple records: 2 records")))

(defun test-array-values ()
  (let* ((input (format nil "~Ctags~Ca~Cb~Cc~C" +stx+ +us+ +gs+ +gs+ +etx+))
         (doc (parse input))
         (tags (get-value (first (document-records doc)) "tags")))
    (assert-eq (length (document-records doc)) 1 "Array: 1 record")
    (assert-true (listp tags) "Array: tags is list")
    (assert-eq (length tags) 3 "Array: 3 elements")))

(defun test-header ()
  (let* ((input (format nil "~Chsv~C1.0~Ctype~Cusers~Cname~CAlice~C"
                        +soh+ +us+ +rs+ +us+ +stx+ +us+ +etx+))
         (doc (parse input)))
    (assert-true (not (null (document-header doc))) "Header: has header")
    (assert-eq (get-value (document-header doc) "hsv") "1.0" "Header: hsv=1.0")
    (assert-eq (get-value (document-header doc) "type") "users" "Header: type=users")
    (assert-eq (length (document-records doc)) 1 "Header: 1 record")))

(defun test-nesting ()
  (let* ((input (format nil "~Cuser~C~Cname~CAlice~Cemail~Ca@b.com~C~C"
                        +stx+ +us+ +so+ +us+ +rs+ +us+ +si+ +etx+))
         (doc (parse input))
         (user (get-value (first (document-records doc)) "user")))
    (assert-eq (length (document-records doc)) 1 "Nesting: 1 record")
    (assert-true (hash-table-p user) "Nesting: user is object")
    (assert-eq (get-value user "name") "Alice" "Nesting: name=Alice")
    (assert-eq (get-value user "email") "a@b.com" "Nesting: email=a@b.com")))

(defun test-deep-nesting ()
  (let* ((input (format nil "~Cdata~C~Clevel1~C~Clevel2~Cdeep~C~C~C"
                        +stx+ +us+ +so+ +us+ +so+ +us+ +si+ +si+ +etx+))
         (doc (parse input))
         (data (get-value (first (document-records doc)) "data"))
         (level1 (get-value data "level1")))
    (assert-eq (length (document-records doc)) 1 "Deep nesting: 1 record")
    (assert-eq (get-value level1 "level2") "deep" "Deep nesting: level2=deep")))

(defun test-newlines ()
  (let* ((input (format nil "~Ctext~Cline1~Cline2~Cline3~C"
                        +stx+ +us+ #\Newline #\Newline +etx+))
         (doc (parse input))
         (expected (format nil "line1~Cline2~Cline3" #\Newline #\Newline)))
    (assert-eq (length (document-records doc)) 1 "Newlines: 1 record")
    (assert-eq (get-value (first (document-records doc)) "text") expected "Newlines: preserved")))

(defun test-quotes ()
  (let* ((input (format nil "~Cmsg~CHe said \"hello\"~C" +stx+ +us+ +etx+))
         (doc (parse input)))
    (assert-eq (length (document-records doc)) 1 "Quotes: 1 record")
    (assert-eq (get-value (first (document-records doc)) "msg")
               "He said \"hello\"" "Quotes: no escaping")))

(defun test-mixed-content ()
  (let* ((input (format nil "ignored~Cname~CAlice~Calso ignored" +stx+ +us+ +etx+))
         (doc (parse input)))
    (assert-eq (length (document-records doc)) 1 "Mixed: 1 record")
    (assert-eq (get-value (first (document-records doc)) "name") "Alice" "Mixed: name=Alice")))

(defun test-multiple-blocks ()
  (let* ((input (format nil "~Ca~C1~Cjunk~Cb~C2~C" +stx+ +us+ +etx+ +stx+ +us+ +etx+))
         (doc (parse input)))
    (assert-eq (length (document-records doc)) 2 "Multiple blocks: 2 records")))

(defun test-nested-array ()
  (let* ((input (format nil "~Cuser~C~Cname~CAlice~Ctags~Cadmin~Cuser~C~C"
                        +stx+ +us+ +so+ +us+ +rs+ +us+ +gs+ +si+ +etx+))
         (doc (parse input))
         (user (get-value (first (document-records doc)) "user"))
         (tags (get-value user "tags")))
    (assert-eq (length (document-records doc)) 1 "Nested array: 1 record")
    (assert-eq (get-value user "name") "Alice" "Nested array: name=Alice")
    (assert-true (listp tags) "Nested array: tags is list")
    (assert-eq (length tags) 2 "Nested array: 2 tags")))

(defun run-tests ()
  (setf *passed* 0)
  (setf *failed* 0)

  (format t "==================================================~%")
  (format t "HSV Parser Tests (Common Lisp)~%")
  (format t "==================================================~%")

  (test-basic)
  (test-multiple-records)
  (test-array-values)
  (test-header)
  (test-nesting)
  (test-deep-nesting)
  (test-newlines)
  (test-quotes)
  (test-mixed-content)
  (test-multiple-blocks)
  (test-nested-array)

  (format t "==================================================~%")
  (format t "~D passed, ~D failed~%" *passed* *failed*)
  (format t "==================================================~%")

  (if (> *failed* 0)
      (sb-ext:exit :code 1)
      (sb-ext:exit :code 0)))

(run-tests)
